//+------------------------------------------------------------------+
//|                                  MACD_Momentum_&_Trend_Index.mq4 |
//+------------------------------------------------------------------+
#property copyright   "Indicator by KHTauon"
#property link        "khtauon@protonmail.com"
#property description "Moving Averages based on AllAverages indicator by Igor Durkin (igorad)"
#property description "igorad2003@yahoo.co.uk"
//                     https://newdigitallife.wordpress.com/
#property version     "1.031"

#property indicator_separate_window
#property indicator_buffers 9
#property indicator_color1  Goldenrod
#property indicator_color2  SaddleBrown
#property indicator_color3  DarkSlateBlue
#property indicator_color4  MediumSlateBlue
#property indicator_color5  Red
#property indicator_color6  Maroon
#property indicator_color7  Green
#property indicator_color8  Lime
#property indicator_color9  Aquamarine
#property strict 

enum ENUM_MA_MODE
{
   SMA,                 // Simple Moving Average
   SMMA,                // Smoothed Moving Average
   LWMA,                // Linear Weighted Moving Average
   ExpWMA,              // Exponential Weighted Moving Average
   PWMA,                // Parabolic Weighted Moving Average
   SineWMA,             // Sine Weighted Moving Average   
   TriMA,               // Triangular Moving Average
   EMA,                 // Exponential Moving Average
   REMA,                // Regularized Exponential Moving Average by C.Satchwell
   FEMA,                // Fast Exponential Moving Average
   LEMA,                // Leader Exponential Moving Average
   MEMA,                // McNicholl Exponential Moving Average
   Wilder,              // Wilder Exponential Moving Average
   DsEMA,               // Double Smoothed Exponential Moving Average
   TsEMA,               // Triple Smoothed Exponential Moving Average
   DEMA,                // Double Exponential Moving Average by P.Mulloy
   TEMA,                // Triple Exponential Moving Average by P.Mulloy
   ZLEMA,               // Zero-Lag Exponential Moving Average by J.Ehlers   
   ZLDEMA,              // Zero-Lag Double Exponential Moving Average
   ZLTEMA,              // Zero-Lag Triple Exponential Moving Average
   T3_ori,              // T3 by T.Tillson (Original Version)
   T3,                  // T3 by T.Tillson (Normalized Period)
   Median,              // Moving Median
   GeoMean,             // Geometric Mean
   LSMA,                // Least Square Moving Average (or Linear Regression Line)   
   ILRS,                // Integral of Linear Regression Slope
   IE_2,                // Combination of LSMA and ILRS (IE2)
   RMT,                 // Recursive Moving Trendline
   FRAMA,               // Fractal Adaptive Moving Average
   ALXMA,               // Alexander Moving Average
   NLMA,                // Non-Lag Moving Average by Igor Durkin
   HMA,                 // Hull Moving Average by A.Hull
   ALMA,                // Arnaud Legoux Moving Average
   JSmooth,             // Simple Jurik Smoothing
   MD,                  // McGinley Dynamic
   ITrend,              // Instantaneous Trendline by J.Ehlers   
   Laguerre,            // Laguerre Filter by J.Ehlers
   BF2P,                // Two-Pole Butterworth Filter by J.Ehlers
   BF3P,                // Three-Pole Butterworth Filter by J.Ehlers
   SSm2P,               // Two-Pole SuperSmoother by J.Ehlers
   SSm3P,               // Three-Pole SuperSmoother by J.Ehlers
   Decycler,            // Simple Decycler by J.Ehlers   
   VWMA,                // Volume-Weighted Moving Average
   VEMA,                // Volume-Weighted Exponential Moving Average
   eVWMA,               // Elastic Volume-Weighted Moving Average
};

enum ENUM_PRICE
{
   close,               // Close
   open,                // Open
   high,                // High
   low,                 // Low
   median,              // Median
   typical,             // Typical
   weightedClose,       // Weighted Close
   medianBody,          // Median Body (Open+Close)/2
   average,             // Average (High+Low+Open+Close)/4
   trendBiased,         // Trend Biased
   trendBiasedExt,      // Trend Biased(extreme)
   haClose,             // Heiken Ashi Close
   haOpen,              // Heiken Ashi Open
   haHigh,              // Heiken Ashi High   
   haLow,               // Heiken Ashi Low
   haMedian,            // Heiken Ashi Median
   haTypical,           // Heiken Ashi Typical
   haWeighted,          // Heiken Ashi Weighted Close
   haMedianBody,        // Heiken Ashi Median Body
   haAverage,           // Heiken Ashi Average
   haTrendBiased,       // Heiken Ashi Trend Biased
   haTrendBiasedExt     // Heiken Ashi Trend Biased(extreme)   
};

enum ENUM_DISPLAY
{
   mom,      // MACD Momentum
   macd,     // MACD Trend
};

#define pi 3.14159265358979323846

//
//
//
 
input  ENUM_DISPLAY   D_Type        = mom;        // Display Mode
input  ENUM_MA_MODE   MA_Method_1   = EMA;        // Fast MA Method
input  ENUM_MA_MODE   MA_Method_2   = EMA;        // Slow MA Method
input  ENUM_PRICE     Price1        = 0;          // Price to use for Fast MA
input  ENUM_PRICE     Price2        = 0;          // Price to use for Slow MA
extern int            MA_Period_1   = 12;         // Fast MA Period
input  int            MA_Period_2   = 26;         // Slow MA Period 
input  int            V_Period      = 5;          // Momentum Lookback Period
input  string         Note1         = "-----";    // -----Filter Line Settings-----
input  bool           F_Line        = true;       // Display Filter Line
input  double         F_Multip_1    = 1.0;        // MACD Filter Multiplier
input  ENUM_MA_MODE   F_MA_Method_1 = EMA;        // MACD Filter MA Method
input  int            F_Period_1    = 9;          // MACD Filter MA Period
input  double         F_Multip_2    = 1.0;        // Momentum Filter Multiplier
input  ENUM_MA_MODE   F_MA_Method_2 = EMA;        // Momentum Filter MA Method
input  int            F_Period_2    = 3;          // Momentum Filter MA Period
input  string         Note2         = "-----";    // -----Width Settings-----
input  int            Bar_Width     = 3;          // Bar Width
input  int            Dot_Width     = 3;          // Dot Width
input  int            Filter_Width  = 1;          // Filter Line Width

double iprice1[],iprice2[],MA1[],MA2[],macd[],mom[],filter1[],filter2[];
double buffer1[],buffer2[],buffer3[],buffer4[],buffer5[],buffer6[],buffer7[],buffer8[],filter[];
int    masize1, masize2, masize3,masize4, maxmasize;
string vname;

//+------------------------------------------------------------------+
//| Custom indicator initialization function                         |
//+------------------------------------------------------------------+
int OnInit()
{
   
   if(D_Type == 0) {
      vname = "Momentum"; IndicatorShortName("MACD MTI (Momentum)");
   } else {
      vname = "MACD";     IndicatorShortName("MACD MTI (Trend)");
   }
//---- 
   IndicatorBuffers(17);
   IndicatorDigits(Digits);
   SetIndexStyle(0, DRAW_HISTOGRAM, STYLE_SOLID, Bar_Width);
   SetIndexBuffer(0, buffer1); SetIndexLabel(0, vname);
   SetIndexStyle(1, DRAW_HISTOGRAM, STYLE_SOLID, Bar_Width);
   SetIndexBuffer(1, buffer2); SetIndexLabel(1, vname);
   SetIndexStyle(2, DRAW_HISTOGRAM, STYLE_SOLID, Bar_Width);
   SetIndexBuffer(2, buffer3); SetIndexLabel(2, vname);
   SetIndexStyle(3, DRAW_HISTOGRAM, STYLE_SOLID, Bar_Width);
   SetIndexBuffer(3, buffer4); SetIndexLabel(3, vname);
   SetIndexStyle(4, DRAW_ARROW, EMPTY, Dot_Width);
   SetIndexBuffer(4, buffer5); SetIndexArrow(4,159);
   SetIndexEmptyValue(4,EMPTY_VALUE); SetIndexLabel(4,NULL);
   SetIndexStyle(5, DRAW_ARROW, EMPTY, Dot_Width);
   SetIndexBuffer(5, buffer6); SetIndexArrow(5,159);
   SetIndexEmptyValue(5,EMPTY_VALUE); SetIndexLabel(5,NULL);
   SetIndexStyle(6, DRAW_ARROW, EMPTY, Dot_Width);
   SetIndexBuffer(6, buffer7); SetIndexArrow(6,159);
   SetIndexEmptyValue(6,EMPTY_VALUE); SetIndexLabel(6,NULL);
   SetIndexStyle(7, DRAW_ARROW, EMPTY, Dot_Width);
   SetIndexBuffer(7, buffer8); SetIndexArrow(7,159);
   SetIndexEmptyValue(7,EMPTY_VALUE); SetIndexLabel(7,NULL);
   SetIndexStyle(8, DRAW_LINE, STYLE_SOLID, Filter_Width);
   SetIndexBuffer(8, filter); SetIndexLabel(8, "Filter");
   SetIndexBuffer(9, filter1); 
   SetIndexBuffer(10, filter2);
   SetIndexBuffer(11, iprice1); 
   SetIndexBuffer(12, iprice2);
   SetIndexBuffer(13, MA1);
   SetIndexBuffer(14, MA2);
   SetIndexBuffer(15, macd);
   SetIndexBuffer(16, mom);
//---- 
   masize1 = averageSize(MA_Method_1);
   masize2 = averageSize(MA_Method_2);
   masize3 = averageSize(F_MA_Method_1);
   masize4 = averageSize(F_MA_Method_2);
   maxmasize = MathMax(MathMax(masize3,masize4),MathMax(masize1,masize2));
//----   
   if(maxmasize > 0) ArrayResize(matpm,maxmasize);
      
   return(INIT_SUCCEEDED);
}


//+------------------------------------------------------------------+
//| Custom indicator iteration function                              |
//+------------------------------------------------------------------+
int OnCalculate(const int rates_total,
                const int prev_calculated,
                const datetime &time[],
                const double &open[],
                const double &high[],
                const double &low[],
                const double &close[],
                const long &tick_volume[],
                const long &volume[],
                const int &spread[])
{
   int i;
   int limit = rates_total-prev_calculated;
   if(limit>=rates_total) limit = rates_total-1;

   for(i=limit; i>=0; i--) {

      iprice1[i] = (Price1 <= 10) ? getPrice((int)Price1,i) : HeikenAshi(0,Price1-11,Bars-1,i);
      iprice2[i] = (Price2 <= 10) ? getPrice((int)Price2,i) : HeikenAshi(1,Price2-11,Bars-1,i);

      if(i > Bars-MathMax(MA_Period_1,MA_Period_2)-1) continue;

      MA1[i]  = allAveragesOnArray(0,iprice1,MA_Period_1,(int)MA_Method_1,masize1,Bars-MathMax(MA_Period_1,MA_Period_2)-1,i);
      MA2[i]  = allAveragesOnArray(1,iprice2,MA_Period_2,(int)MA_Method_2,masize2,Bars-MathMax(MA_Period_1,MA_Period_2)-1,i);

      macd[i] = MA1[i]-MA2[i];
      mom[i]  = 2*macd[i]-macd[ArrayMaximum(macd,V_Period,i)]-macd[ArrayMinimum(macd,V_Period,i)];

      if(i > Bars-MathMax(MA_Period_1,MA_Period_2)-F_Period_2-V_Period-1) continue;

      filter1[i] = F_Multip_1*allAveragesOnArray(2,macd,F_Period_1,(int)F_MA_Method_1,masize3,Bars-MathMax(MA_Period_1,MA_Period_2)-F_Period_1-1,i);
      filter2[i] = F_Multip_2*allAveragesOnArray(3,mom ,F_Period_2,(int)F_MA_Method_2,masize3,Bars-MathMax(MA_Period_1,MA_Period_2)-F_Period_2-V_Period-1,i);
      filter[i]  = (F_Line == 1)? (D_Type == 0) ? filter2[i] : filter1[i] : EMPTY_VALUE;

      buffer1[i] = EMPTY_VALUE; buffer2[i] = EMPTY_VALUE; buffer3[i] = EMPTY_VALUE; buffer4[i] = EMPTY_VALUE; 
      buffer5[i] = EMPTY_VALUE; buffer6[i] = EMPTY_VALUE; buffer7[i] = EMPTY_VALUE; buffer8[i] = EMPTY_VALUE; 

      if(mom[i]<=filter2[i] && mom[i]<=0) (D_Type == 0) ? buffer1[i] = mom[i] : buffer5[i] = 0;
      if(mom[i]> filter2[i] && mom[i]<=0) (D_Type == 0) ? buffer2[i] = mom[i] : buffer6[i] = 0;
      if(mom[i]<=filter2[i] && mom[i]> 0) (D_Type == 0) ? buffer3[i] = mom[i] : buffer7[i] = 0;
      if(mom[i]> filter2[i] && mom[i]> 0) (D_Type == 0) ? buffer4[i] = mom[i] : buffer8[i] = 0;

      if(macd[i]<=filter1[i] && macd[i]<=0) (D_Type == 1) ? buffer1[i] = macd[i] : buffer5[i] = 0;
      if(macd[i]> filter1[i] && macd[i]<=0) (D_Type == 1) ? buffer2[i] = macd[i] : buffer6[i] = 0;
      if(macd[i]<=filter1[i] && macd[i]> 0) (D_Type == 1) ? buffer3[i] = macd[i] : buffer7[i] = 0;
      if(macd[i]> filter1[i] && macd[i]> 0) (D_Type == 1) ? buffer4[i] = macd[i] : buffer8[i] = 0;

   }

   return(rates_total);
}

//
//
//
//

int averageSize(int mode)
{   
   int arraysize;
   
   switch(mode)
   {
   case DEMA     : arraysize = 2; break;
   case T3_ori   : arraysize = 6; break;
   case JSmooth  : arraysize = 5; break;
   case TEMA     : arraysize = 3; break;
   case T3       : arraysize = 6; break;
   case Laguerre : arraysize = 4; break;
   case DsEMA    : arraysize = 2; break;
   case TsEMA    : arraysize = 3; break;
   case MEMA     : arraysize = 2; break;
   case VEMA     : arraysize = 2; break;
   case RMT      : arraysize = 2; break;
   case LEMA     : arraysize = 2; break;
   case ZLDEMA   : arraysize = 4; break;
   case ZLTEMA   : arraysize = 6; break;
   default       : arraysize = 0; break;
   }
   
   return(arraysize);
}


double   matpm[][4][2], ma[4][4];
datetime prevtime[4];  

double allAveragesOnArray(int index,double& price[],int period,int mode,int arraysize,int cbars,int bar)
{
   int i;
   double MA[4];
        
   switch(mode) 
	{
	case EMA: case FEMA: case Wilder: case SMMA: case ZLEMA: case DEMA: case T3_ori: case ITrend: case REMA: case JSmooth: case TEMA: case T3: case Laguerre: case MD: case FRAMA:
	case BF2P: case BF3P: case SSm2P: case SSm3P: case Decycler: case eVWMA: case DsEMA: case TsEMA: case MEMA: case VEMA: case RMT: case LEMA: case ZLDEMA: case ZLTEMA: 
		
		if(prevtime[index] != Time[bar])
      {
      ma[index][3] = ma[index][2]; 
      ma[index][2] = ma[index][1]; 
      ma[index][1] = ma[index][0]; 
   
      if(arraysize > 0) for(i=0;i<arraysize;i++) matpm[i][index][1] = matpm[i][index][0];
      prevtime[index] = Time[bar]; 
      }
   
      if(mode == ITrend || mode == REMA || (mode >= BF2P && mode < eVWMA)) for(i=0;i<4;i++) MA[i] = ma[index][i]; 
	}
      
   switch(mode)
   {
   case SMMA      : ma[index][0] = SMMAOnArray(price,ma[index][1],period,cbars,bar); break;
   case LWMA      : ma[index][0] = LWMAOnArray(price,period,bar); break;
   case PWMA      : ma[index][0] = PWMAOnArray(price,2,period,bar); break;
   case SineWMA   : ma[index][0] = SineWMAOnArray(price,period,bar); break;   
   case TriMA     : ma[index][0] = TriMAOnArray(price,period,cbars,bar); break;  
   case EMA       : ma[index][0] = EMAOnArray(price[bar],ma[index][1],period,cbars,bar); break;
   case REMA      : ma[index][0] = REMAOnArray(price[bar],MA,period,0.5,cbars,bar); break;   
   case FEMA      : ma[index][0] = FEMAOnArray(price[bar],ma[index][1],period,cbars,bar); break;
   case LEMA      : ma[index][0] = LEMAOnArray(index,price[bar],period,cbars,bar); break;
   case MEMA      : ma[index][0] = MEMAOnArray(index,price[bar],period,cbars,bar); break;
   case Wilder    : ma[index][0] = WilderOnArray(price[bar],ma[index][1],period,cbars,bar); break;  
   case DsEMA     : ma[index][0] = DsEMAOnArray(index,price[bar],period,cbars,bar); break;
   case TsEMA     : ma[index][0] = TsEMAOnArray(index,price[bar],period,cbars,bar); break;
   case DEMA      : ma[index][0] = DEMAOnArray(index,0,price[bar],period,1,cbars,bar); break;
   case TEMA      : ma[index][0] = TEMAOnArray(index,0,price[bar],period,1,cbars,bar); break;
   case ZLEMA     : ma[index][0] = ZLEMAOnArray(price,ma[index][1],period,cbars,bar); break;
   case ZLDEMA    : ma[index][0] = ZLDEMAOnArray(index,0,price[bar],period,1,cbars,bar); break;
   case ZLTEMA    : ma[index][0] = ZLTEMAOnArray(index,0,price[bar],period,1,cbars,bar); break;
   case T3_ori    : ma[index][0] = T3_oriOnArray(index,0,price[bar],period,0.7,cbars,bar); break;
   case T3        : ma[index][0] = T3OnArray(index,0,price[bar],period,0.7,cbars,bar); break;   
   case Median    : ma[index][0] = MedianOnArray(price,period,cbars,bar); break;
   case GeoMean   : ma[index][0] = GeoMeanOnArray(price,period,cbars,bar); break;   
   case LSMA      : ma[index][0] = LSMAOnArray(price,period,bar); break;
   case ILRS      : ma[index][0] = ILRSOnArray(price,period,cbars,bar); break;
   case IE_2      : ma[index][0] = IE2OnArray(price,period,cbars,bar); break;
   case RMT       : ma[index][0] = RMTOnArray(index,price[bar],period,cbars,bar); break;      
   case FRAMA     : ma[index][0] = FRAMAOnArray(price,ma[index][1],period,cbars,bar); break;
   case ALXMA     : ma[index][0] = ALXMAOnArray(price,period,bar); break;
   case HMA       : ma[index][0] = HMAOnArray(price,period,cbars,bar); break;
   case NLMA      : ma[index][0] = NLMAOnArray(price,period,cbars,bar); break;
   case ALMA      : ma[index][0] = ALMAOnArray(price,period,0.85,8,bar); break;
   case JSmooth   : ma[index][0] = JSmoothOnArray(index,0,price[bar],period,1,cbars,bar); break;
   case MD        : ma[index][0] = McGinleyOnArray(price[bar],ma[index][1],period,cbars,bar); break;
   case ITrend    : ma[index][0] = ITrendOnArray(price,MA,period,cbars,bar); break;
   case Laguerre  : ma[index][0] = LaguerreOnArray(index,price[bar],period,4,cbars,bar); break;
   case BF2P      : ma[index][0] = BF2POnArray(price,MA,period,cbars,bar); break;
   case BF3P      : ma[index][0] = BF3POnArray(price,MA,period,cbars,bar); break;
   case SSm2P     : ma[index][0] = SSm2POnArray(price,MA,period,cbars,bar); break;
   case SSm3P     : ma[index][0] = SSm3POnArray(price,MA,period,cbars,bar); break;
   case Decycler  : ma[index][0] = DecyclerOnArray(price,MA,period,cbars,bar); return(price[bar] - ma[index][0]); 
   case VWMA      : ma[index][0] = VWMAOnArray(price,period,bar); break;
   case VEMA      : ma[index][0] = VEMAOnArray(index,price[bar],period,cbars,bar); break;
   case eVWMA     : ma[index][0] = eVWMAOnArray(price[bar],ma[index][1],period,cbars,bar); break;
   default        : ma[index][0] = SMAOnArray(price,period,bar); break;
   }
   
   return(ma[index][0]);
}

// SMA - Simple Moving Average
double SMAOnArray(double& array[],int per,int bar)
{
   double sum = 0;
   for(int i=0;i<per;i++) sum += array[bar+i];
   
   return(sum/per);
}                
// EMA - Exponential Moving Average
double EMAOnArray(double price,double prev,int per,int cbars,int bar)
{
   double ema;
   
   if(bar >= cbars)
      ema = price;
   else 
      ema = prev + 2.0/(1 + per)*(price - prev); 
   
   return(ema);
}
// FEMA - Fast Exponential Moving Average
double FEMAOnArray(double price,double prev,int per,int cbars,int bar)
{
   double fema, alpha = 2.0/(2.0+(per-1.0)/2.0);
   
   if(bar >= cbars)
      fema = price;
   else 
      fema = prev + alpha * (price - prev); 
   
   return(fema);
}
// Wilder - Wilder Exponential Moving Average
double WilderOnArray(double price,double prev,int per,int cbars,int bar)
{
   double wilder;
   
   if(bar >= cbars)
      wilder = price; 
   else 
      wilder = prev + (price - prev)/per; 
   
   return(wilder);
}
// LWMA - Linear Weighted Moving Average 
double LWMAOnArray(double& array[],int per,int bar)
{
   double sum = 0, weight = 0;
   
   for(int i=0;i<per;i++)
   { 
      weight += (per - i);
      sum    += array[bar+i]*(per - i);
   }
   
   if(weight > 0) return(sum/weight); else return(0); 
}
// FRAMA - Fractal Adaptive Moving Average 
double FRAMAOnArray(double& array[],double prev,int per,int cbars,int bar)
{
   double frama, Hi1,Hi2,Hi3,Lo1,Lo2,Lo3,D=1,alpha=0;
   
   if(bar >= cbars)
      frama = array[bar];
   else 
   {   
      Hi1 = array[ArrayMaximum(array,per,bar)];
      Lo1 = array[ArrayMinimum(array,per,bar)];
      Hi2 = array[ArrayMaximum(array,per,bar+per)];
      Lo2 = array[ArrayMinimum(array,per,bar+per)];
      Hi3 = array[ArrayMaximum(array,2*per,bar)];
      Lo3 = array[ArrayMinimum(array,2*per,bar)];

      double N1 = (Hi1-Lo1)/per;
      double N2 = (Hi2-Lo2)/per;
      double N3 = (Hi3-Lo3)/(2.0*per);

      D     = (MathLog(N1+N2)-MathLog(N3))/MathLog(2.0);
      alpha = MathExp(-4.6*(D-1.0));
     
      frama = alpha * array[bar] + (1 - alpha) * prev;
   }
   
   return(frama);
}
// SineWMA - Sine Weighted Moving Average
double SineWMAOnArray(double& array[],int per,int bar)
{
   double sum = 0, weight = 0;
  
   for(int i=0;i<per;i++)
   { 
      weight += MathSin(pi*(i + 1)/(per + 1));
      sum    += array[bar+i]*MathSin(pi*(i + 1)/(per + 1)); 
   }
   
   if(weight > 0) return(sum/weight); else return(0); 
}
// LSMA - Least Square Moving Average (or EPMA, Linear Regression Line)
double LSMAOnArray(double& array[],int per,int bar)
{   
   double sum = 0;
   
   for(int i=per;i>=1;i--)
      sum += (i - (per + 1)/3.0)*array[bar+per-i];
   
   return(sum*6/(per*(per + 1)));
}
// SMMA - Smoothed Moving Average
double SMMAOnArray(double& array[],double prev,int per,int cbars,int bar)
{
   double smma;
   
   if(bar >= cbars)
      smma = array[bar];
   else
   {
      double sum = 0;
      for(int i=0;i<per;i++) sum += array[bar+i+1];
      smma = (sum - prev + array[bar])/per;
   }
   
   return(smma);
}                
// HMA - Hull Moving Average by Alan Hull
double HMAOnArray(double& array[],int per,int cbars,int bar)
{
   double hma = 0, _matpm[];
   int len = (int)MathSqrt(per);
   
   ArrayResize(_matpm,len);
   
   if(bar == cbars)
      hma = array[bar]; 
   else if(bar  < cbars)
   {
      for(int i=0;i<len;i++) _matpm[i] = 2*LWMAOnArray(array,per/2,bar+i) - LWMAOnArray(array,per,bar+i);  
      hma = LWMAOnArray(_matpm,len,0); 
   }  

   return(hma);
}
// ZLEMA - Zero-Lag Exponential Moving Average
double ZLEMAOnArray(double& price[],double prev,int per,int cbars,int bar)
{
   int lag = (int)(0.5*(per - 1)); 
   double zema, alpha = 2.0/(1 + (double)per); 
      
   if(bar >= cbars)
      zema = price[bar];
   else 
      zema = alpha*(2*price[bar] - price[bar+lag]) + (1 - alpha)*prev;
   
   return(zema);
}
// DEMA - Double Exponential Moving Average by Patrick Mulloy
double DEMAOnArray(int index,int num,double price,double per,double v,int cbars,int bar)
{
   double dema, alpha = 2.0/(1 + per);
   
   if(bar >= cbars) 
   {
      dema                 = price; 
      matpm[num  ][index][0] = dema; 
      matpm[num+1][index][0] = dema;
   }
   else
   {
      matpm[num  ][index][0] = matpm[num  ][index][1] + alpha*(price              - matpm[num  ][index][1]); 
      matpm[num+1][index][0] = matpm[num+1][index][1] + alpha*(matpm[num][index][0] - matpm[num+1][index][1]); 
      dema                 = matpm[num  ][index][0] * (1+v) - matpm[num+1][index][0] * v;
   }
   
   return(dema);
}
// T3 by T.Tillson
double T3_oriOnArray(int index,int num,double price,int per,double v,int cbars,int bar)
{
   double dema1, dema2, T3;
   
   if(bar >= cbars) 
   {
      T3 = price; 
      for(int k=0;k<6;k++) matpm[num+k][index][0] = price;
   }
   else  
   {
      dema1 = DEMAOnArray(index,num  ,price,per,v,cbars,bar); 
      dema2 = DEMAOnArray(index,num+2,dema1,per,v,cbars,bar); 
      T3    = DEMAOnArray(index,num+4,dema2,per,v,cbars,bar);
   }
   
   return(T3);
}
// ZLDEMA - Zero Lag DEMA
double ZLDEMAOnArray(int index,int num,double price,int per,double v,int cbars,int bar)
{
   double dema1, dema2, zldema;
   
   if(bar >= cbars) 
   {
      zldema = price; 
      for(int k=0;k<4;k++) matpm[num+k][index][0] = price;
   }
   else 
   {
      dema1  = DEMAOnArray(index,num  ,price,per,v,cbars,bar); 
      dema2  = DEMAOnArray(index,num+2,dema1,per,v,cbars,bar); 
      zldema = 2.0 * dema1 - dema2;
   }
   
   return(zldema);
}
// ITrend - Instantaneous Trendline by J.Ehlers
double ITrendOnArray(double& price[],double& array[],int per,int cbars,int bar)
{
   double it, alpha = 2.0/(per + 1);

   if(bar < cbars && array[1] > 0 && array[2] > 0) 
      it =  (alpha - 0.25*alpha*alpha)*price[bar] + 0.5*alpha*alpha*price[bar+1] 
           -(alpha - 0.75*alpha*alpha)*price[bar+2] + 2*(1 - alpha)*array[1] 
           -(1 - alpha)*(1 - alpha)*array[2];
   else 
      it = (price[bar] + 2*price[bar+1] + price[bar+2])/4;
 
   return(it);
}
// Median - Moving Median
double MedianOnArray(double& price[],int per,int cbars,int bar)
{
   double median, array[];
   ArrayResize(array,per);
   
   if(bar <= cbars)
   {
      for(int i=0;i<per;i++) array[i] = price[bar+i];
      ArraySort(array,WHOLE_ARRAY,0,MODE_DESCEND);
      
      int num = (int)MathRound((per - 1)*0.5);
      
      if(MathMod(per,2) > 0) median = array[num]; 
      else median = 0.5*(array[num] + array[num+1]);
   }
   else
      median = price[bar];
    
   return(median); 
}
// GeoMean - Geometric Mean
double GeoMeanOnArray(double& price[],int per,int cbars,int bar)
{
   double gmean = 0;
   
   if(bar >= cbars)
      gmean = price[bar];
   else
   {
      gmean = MathPow(price[bar],1.0/per); 
      for(int i=1;i<per;i++) gmean *= MathPow(price[bar+i],1.0/per);
   }
   
   return(gmean);
}
// REMA - Regularized EMA by Chris Satchwell 
double REMAOnArray(double price,double& array[],int per,double lambda,int cbars,int bar)
{
   double rema, alpha =  2.0/(per + 1);
   
   if(bar < cbars  && array[1] > 0 && array[2] > 0) 
      rema = (array[1]*(1 + 2*lambda) + alpha*(price - array[1]) - lambda*array[2])/(1 + lambda); 
   else 
      rema = price; 
   
   return(rema);
}
// ILRS - Integral of Linear Regression Slope 
double ILRSOnArray(double& price[],int per,int cbars,int bar)
{
   double slope, sum  = per*(per - 1)*0.5;
   double ilrs = 0, sum2 = (per - 1)*per*(2*per - 1)/6.0;
   
   if(bar < cbars)
   {  
   double sum1 = 0;
   double sumy = 0;
      for(int i=0;i<per;i++)
      { 
      sum1 += i*price[bar+i];
      sumy += price[bar+i];
      }
   double num1 = per*sum1 - sum*sumy;
   double num2 = sum*sum - per*sum2;
   
   if(num2 != 0) slope = num1/num2; else slope = 0; 
   ilrs = slope + SMAOnArray(price,per,bar);
   }
   
   return(ilrs);
}
// IE/2 - Combination of LSMA and ILRS 
double IE2OnArray(double& price[],int per,int cbars,int bar)
{
   double ie;
   
   if(bar < cbars) 
      ie = 0.5*(ILRSOnArray(price,per,cbars,bar) + LSMAOnArray(price,per,bar));
   else
      ie = price[bar];
      
   return(ie); 
}
 
// TriMA - Triangular Moving Average 
double TriMAOnArray(double& array[],int per,int cbars,int bar)
{
   int len1 = (int)MathFloor((per + 1)*0.5);
   int len2 = (int)MathCeil ((per + 1)*0.5);
   double sum = 0;
   
   if(bar < cbars)
       for(int i = 0;i < len2;i++) sum += SMAOnArray(array,len1,bar+i);
   
   return(sum/len2);
}
// VWMA - Volume Weighted Moving Average 
double VWMAOnArray(double& array[],int per,int bar)
{
   double sum = 0, weight = 0, maxvol = 0, vwma = 0;
   
   for(int i=0;i<per;i++)
   { 
      weight += (double)Volume[bar+i];
      sum    += array[bar+i]*Volume[bar+i];
   }
     
   if(weight > 0) return(sum/weight); else return(0); 
} 
// JSmooth - Smoothing by Mark Jurik
double JSmoothOnArray(int index,int num,double price,int per,double power,int cbars,int bar)
{
   double beta  = 0.45*(per - 1)/(0.45*(per - 1) + 2);
	double alpha = MathPow(beta,power);
	
	if(bar >= cbars)
	{
	   matpm[num+4][index][0] = price;
	   matpm[num+0][index][0] = price;
	   matpm[num+2][index][0] = price;
	}
	else  
   {
   	matpm[num+0][index][0] = (1 - alpha)*price + alpha*matpm[num+0][index][1];
   	matpm[num+1][index][0] = (price - matpm[num+0][index][0])*(1-beta) + beta*matpm[num+1][index][1];
   	matpm[num+2][index][0] =  matpm[num+0][index][0] + matpm[num+1][index][0];
   	matpm[num+3][index][0] = (matpm[num+2][index][0] - matpm[num+4][index][1])*MathPow((1-alpha),2) + MathPow(alpha,2)*matpm[num+3][index][1];
   	matpm[num+4][index][0] =  matpm[num+4][index][1] + matpm[num+3][index][0]; 
   }
   return(matpm[num+4][index][0]);
}
// ALMA by Arnaud Legoux / Dimitris Kouzis-Loukas / Anthony Cascino
double ALMAOnArray(double& price[],int per,double offset,double sigma,int bar)
{
   double m = MathFloor(offset*(per - 1)), s = per/sigma, w, sum = 0, wsum = 0;		
	
	for(int i=0;i<per;i++) 
	{
   	w     = MathExp(-((i - m)*(i - m))/(2*s*s));
      wsum += w;
      sum  += price[bar+(per-1-i)]*w; 
   }
   
   if(wsum != 0) return(sum/wsum); else return(0);
}   
// TEMA - Triple Exponential Moving Average by Patrick Mulloy
double TEMAOnArray(int index,int num,double price,int per,double v,int cbars,int bar)
{
   double tema, alpha = 2.0/(per+1);
	
	if(bar >= cbars)
	{
   	matpm[0][index][0] = price;
   	matpm[1][index][0] = price;
   	matpm[2][index][0] = price;
   	tema             = price;
	}
	else 
   {
   	matpm[num  ][index][0] = matpm[num  ][index][1] + alpha *(price                - matpm[num  ][index][1]);
   	matpm[num+1][index][0] = matpm[num+1][index][1] + alpha *(matpm[num  ][index][0] - matpm[num+1][index][1]);
   	matpm[num+2][index][0] = matpm[num+2][index][1] + alpha *(matpm[num+1][index][0] - matpm[num+2][index][1]);
   	tema                 = matpm[num  ][index][0] + v*(matpm[num  ][index][0] + v*(matpm[num  ][index][0]-matpm[num+1][index][0]) - matpm[num+1][index][0] - v*(matpm[num+1][index][0] - matpm[num+2][index][0])); 
	}
   
   return(tema);
}
// ZLTEMA - Zero Lag TEMA
double ZLTEMAOnArray(int index,int num,double price,int per,double v,int cbars,int bar)
{
   double tema1, tema2, zltema;
   
   if(bar >= cbars) 
   {
      zltema = price; 
      for(int k=0;k<6;k++) matpm[num+k][index][0] = price;
   }
   else 
   {
      tema1  = TEMAOnArray(index,num  ,price,per,v,cbars,bar); 
      tema2  = TEMAOnArray(index,num+3,tema1,per,v,cbars,bar); 
      zltema = 2.0 * tema1 - tema2;
   }
   
   return(zltema);
}
// T3 by T.Tillson (normalized period) 
double T3OnArray(int index,int num,double price,int per,double v,int cbars,int bar)
{
   double len = MathMax((per + 5.0)/3.0 - 1,1), dema1, dema2, T3;
   
   if(bar >= cbars) 
   {
      T3 = price; 
      for(int k=0;k<6;k++) matpm[num+k][index][0] = T3;
   }
   else 
   {
      dema1 = DEMAOnArray(index,num  ,price,len,v,cbars,bar); 
      dema2 = DEMAOnArray(index,num+2,dema1,len,v,cbars,bar); 
      T3    = DEMAOnArray(index,num+4,dema2,len,v,cbars,bar);
   }
      
   return(T3);
}
// Laguerre Filter by J.Ehlers
double LaguerreOnArray(int index,double price,int per,int order,int cbars,int bar)
{
   double gamma = 1 - 10.0/(per + 9);
   double aPrice[];
   
   ArrayResize(aPrice,order);
   
   for(int i=0;i<order;i++)
   {
      if(bar >= cbars) matpm[i][index][0] = price;
      else
      {
         if(i == 0) matpm[i][index][0] = (1 - gamma)*price + gamma*matpm[i][index][1];
         else
         matpm[i][index][0] = -gamma * matpm[i-1][index][0] + matpm[i-1][index][1] + gamma * matpm[i][index][1];
      
      aPrice[i] = matpm[i][index][0];
      }
   }
   double laguerre = TriMAOnArray(aPrice,order,cbars,0);  

   return(laguerre);
}
// MD - McGinley Dynamic
double McGinleyOnArray(double price,double prev,int per,int cbars,int bar)
{
   double md;
   
   if(bar < cbars && (prev != 0 || price!=0)) 
      md = prev + (price - prev)/(per*MathPow(price/prev,4)); 
   else
      md = price;

   return(md);
}
// BF2P - Two-pole Butterworth filter
double BF2POnArray(double& price[],double& array[],int per,int cbars,int bar)
{
   double a  = MathExp(-MathSqrt(2.0)*pi/per);
   double b  = 2*a*MathCos(MathSqrt(2.0)*pi/per);
   double c2 = b;
   double c3 = -a*a;
   double c1 = 1 - c2 - c3;
   double bf2p;
   
   if(bar < cbars && array[1] > 0 && array[2] > 0)
      bf2p = c1*(price[bar] + 2*price[bar+1] + price[bar+2])/4 + c2*array[1] + c3*array[2];
   else
      bf2p = (price[bar] + 2*price[bar+1] + price[bar+2])/4;
   
   return(bf2p);
}
// BF3P - Three-pole Butterworth filter
double BF3POnArray(double& price[],double& array[],int per,int cbars,int bar)
{
   double a  = MathExp(-pi/per);
   double b  = 2*a*MathCos(MathSqrt(3.0)*pi/per);
   double c  = a*a;
   double d2 = b + c;
   double d3 = -(c + b*c);
   double d4 = c*c;
   double d1 = 1 - d2 - d3 - d4;
   double bf3p;
   
   if(bar < cbars && array[1] > 0 && array[2] > 0 && array[3] > 0)
      bf3p = d1*(price[bar] + 3*price[bar+1] + 3*price[bar+2] + price[bar+3])/8 + d2*array[1] + d3*array[2] + d4*array[3];
   else
      bf3p = (price[bar] + 3*price[bar+1] + 3*price[bar+2] + price[bar+3])/8;
   
   return(bf3p);
}
// SSm2P - Two-pole Super Smoother filter
double SSm2POnArray(double& price[],double& array[],int per,int cbars,int bar)
{
   double a  = MathExp(-MathSqrt(2.0)*pi/per);
   double b  = 2*a*MathCos(MathSqrt(2.0)*pi/per);
   double c2 = b;
   double c3 = -a*a;
   double c1 = 1 - c2 - c3;
   double ssm2p;
   
   if(bar < cbars && array[1] > 0 && array[2] > 0)
      ssm2p = c1*price[bar] + c2*array[1] + c3*array[2]; 
   else
      ssm2p = price[bar];
   
   return(ssm2p);
}
// SSm3P - Three-pole Super Smoother filter
double SSm3POnArray(double& price[],double& array[],int per,int cbars,int bar)
{
   double a  = MathExp(-pi/per);
   double b  = 2*a*MathCos(MathSqrt(3.0)*pi/per);
   double c  = a*a;
   double d2 = b + c;
   double d3 = -(c + b*c);
   double d4 = c*c;
   double d1 = 1 - d2 - d3 - d4;
   double ssm3p;
   
   if(bar < cbars && array[1] > 0 && array[2] > 0 && array[3] > 0) 
      ssm3p = d1*price[bar] + d2*array[1] + d3*array[2] + d4*array[3];
   else 
      ssm3p = price[bar];
   
   return(ssm3p);
}
// Decycler - Simple Decycler by J.Ehlers
double DecyclerOnArray(double& price[],double& hp[],int per,int cbars,int bar)
{
   double alpha1 = (MathCos(MathSqrt(2.0)*pi/per) + MathSin(MathSqrt(2.0)*pi/per) - 1)/MathCos(MathSqrt(2.0)*pi/per);
  
   if(bar > cbars - 4) return(0);
   
   hp[0] = (1 - alpha1/2)*(1 - alpha1/2)*(price[bar] - 2*price[bar+1] + price[bar+2]) + 2*(1 - alpha1)*hp[1] - (1 - alpha1)*(1 - alpha1)*hp[2];		
       
   return(hp[0]);
}
// eVWMA - Elastic Volume Weighted Moving Average by C.Fries
double eVWMAOnArray(double price,double prev,int per,int cbars,int bar)
{
   double evwma;
 
   if(bar < cbars && prev > 0)
   {
      double vsum = 0;
      for(int i=0;i<per;i++) vsum += (double)Volume[bar+i];
         
      double dif = vsum - Volume[bar];
             
      if(vsum !=0) evwma = (dif*prev + Volume[bar]*price)/(vsum); 
      else evwma = prev;
   }
   else 
      evwma = price;
    
   return(evwma);
}
// MA_Method=ExpWMA - Exponential Weighted Moving Average 
double ExpWMAOnArray(double& array[],int per,int cbars,int bar)
{
   double sum = 0, weight = 0, alpha = 2.0/(1 + per);
   
   if(bar < cbars){
      for(int i=0;i<per;i++)
      { 
      weight += alpha*MathPow(1-alpha,i);
      sum    += array[bar+i]*alpha*MathPow(1-alpha,i);
      }
   }
   
   if(weight > 0) return(sum/weight); else return(0); 
} 
// DsEMA - Double Smoothed Exponential Moving Average
double DsEMAOnArray(int index,double price,int per,int cbars,int bar)
{
   double alpha = 4.0/(per + 3);
   
   if(bar >= cbars) 
   {
      for(int k=0;k<2;k++) matpm[k][index][0] = price;
      return(price);
   }
   else 
   {
      matpm[0][index][0] = matpm[0][index][1] + alpha*(price            - matpm[0][index][1]);
   	matpm[1][index][0] = matpm[1][index][1] + alpha*(matpm[0][index][0] - matpm[1][index][1]);
   }
      
   return(matpm[1][index][0]);
}
// TsEMA - Triple Smoothed Exponential Moving Average
double TsEMAOnArray(int index,double price,int per,int cbars,int bar)
{
   double alpha = 6.0/(per + 5);
   
   if(bar >= cbars) 
   {
      for(int k=0;k<3;k++) matpm[k][index][0] = price;
      return(price);
   }
   else
   {
      matpm[0][index][0] = matpm[0][index][1] + alpha*(price            - matpm[0][index][1]);
   	matpm[1][index][0] = matpm[1][index][1] + alpha*(matpm[0][index][0] - matpm[1][index][1]);
      matpm[2][index][0] = matpm[2][index][1] + alpha*(matpm[1][index][0] - matpm[2][index][1]);
   }
      
   return(matpm[2][index][0]);
}
// MEMA - McNicholl Exponential Moving Average
double MEMAOnArray(int index,double price,int per,int cbars,int bar)
{
   double mema, alpha = 2.0/(per + 1);
   
   if(bar >= cbars) 
   {
      for(int k=0;k<2;k++) matpm[k][index][0] = price;
      mema = price;
   }
   else 
   {
      matpm[0][index][0] = matpm[0][index][1] + alpha*(price            - matpm[0][index][1]);
   	matpm[1][index][0] = matpm[1][index][1] + alpha*(matpm[0][index][0] - matpm[1][index][1]);
   }
   
   mema = ((2 - alpha) * matpm[0][index][0] - matpm[1][index][0]) / (1 - alpha);
      
   return(mema);
}
// VEMA - Volume-Weighted Exponential Moving Average
double VEMAOnArray(int index,double price,int per,int cbars,int bar)
{
   double vema, alpha = 2.0/(per + 1);
   
   if(bar >= cbars) 
   {
      matpm[0][index][0] = (1 - alpha)*price*Volume[bar];
      matpm[1][index][0] = (1 - alpha)*Volume[bar];
   }
   else 
   {
      matpm[0][index][0] = matpm[0][index][1] + alpha*(price*Volume[bar] - matpm[0][index][1]);
   	matpm[1][index][0] = matpm[1][index][1] + alpha*(Volume[bar]       - matpm[1][index][1]);
   }
  
   if(matpm[1][index][0] > 0) 
      vema = matpm[0][index][0]/matpm[1][index][0];
   else
      vema = price;
   
   return(vema);
}
// MA_Method=RMT - Recursive Moving Trendline
double RMTOnArray(int index,double price,int per,int cbars,int bar)
{
   double alpha = 2.0/(per + 1);
   double init1 = price;
   double init2 = price;
   
   if(bar >= cbars) 
   {
      for(int k=0;k<pi*per;k++) 
      {
         init1 = (1 - alpha) * init1 + price;
         init2 = (1 - alpha) * init2 + alpha * price;
      }
      matpm[0][index][0] = init1;
      matpm[1][index][0] = init2;
   }
   else 
   {
      matpm[0][index][0] = (1 - alpha) * matpm[0][index][1] + price;
      matpm[1][index][0] = (1 - alpha) * matpm[1][index][1] + alpha * (price + matpm[0][index][0] - matpm[0][index][1]);
   }
      
   return(matpm[1][index][0]);
}
// MA_Method=LEMA - Leader Exponential Moving Average
double LEMAOnArray(int index,double price,int per,int cbars,int bar)
{
   double lema = price, alpha = 2.0/(per + 1);
   double init1 = price;
   double init2 = price;
   
   if(bar >= cbars) 
   {
      for(int k=0;k<pi*per;k++) {
         init1 = init1 + alpha * (price - init1);
         init2 = init2 + alpha * (price - init1 - init2);
      }
      matpm[0][index][0] = init1;
      matpm[1][index][0] = init2;
   }
   else 
   {
      matpm[0][index][0] = matpm[0][index][1] + alpha*(price                    - matpm[0][index][1]);
   	matpm[1][index][0] = matpm[1][index][1] + alpha*(price - matpm[0][index][0] - matpm[1][index][1]);
   	lema = matpm[0][index][0] + matpm[1][index][0];
   }
      
   return(lema);
}
// MA_Method=ALXMA - Alexander Moving Average 
double ALXMAOnArray(double& array[],int per,int bar)
{
   if(per<4) return(array[bar]);
   double weight = per - 2;
   double sum    = weight * array[bar];
   
   for(int i=1; i<per; i++)
   { 
      weight += (per - i - 2);
      sum    += (per - i - 2) * array[bar+i];
   }
      
   if(weight > 0) return(sum/weight); else return(0); 
}
// MA_Method=PWMA - Parabolic Weighted Moving Average
double PWMAOnArray(double& array[],int ppow, int per,int bar)
{
   double weight = MathPow(per,ppow);
   double sum    = weight * array[bar];
   
   for(int i=1; i<per; i++)
   { 
      weight += MathPow((per - i),ppow);
      sum    += MathPow((per - i),ppow) * array[bar+i];
   }
      
   if(weight > 0) return(sum/weight); else return(0); 
} 
// NLMA - Non-Lag Moving Average by igorad
double NLMAOnArray(double& array[],int per, int cbars, int bar)
{
   int phase = per - 1, len = 4*per + phase, lb;
   double cycle = 4.0, coeff = 3.0*pi, sum = 0, weight = 0, t, beta, g;
   
   if(bar < cbars - len)                        lb = len;
   else if(bar >= (cbars - len) && bar < cbars) lb = cbars - bar;
   else                                         return(0);
   
   for(int i=0;i<lb;i++)
   {
      if(i < phase - 1) t    = 1.0 * i/(phase-1);
      else              t    = 1.0 + (i-phase+1)*(2.0*cycle - 1.0)/(cycle*per - 1.0);             
                        beta = MathCos(pi*t);

      if(t <= 0.5) g = 1.0;
      else         g = 1.0/(coeff*t +1);

      weight += g * beta;
      sum    += g * beta * array[bar+i];
   }

   if(weight > 0) return(sum/weight); else return(0); 
}
//----------//

// Normal Price
double getPrice(int price,int bar)
{
   double close = Close[bar];
   double open  = Open [bar];
   double high  = High [bar];
   double low   = Low  [bar];
   
   switch(price)
   {   
      case  0: return(close); break;
      case  1: return(open ); break;
      case  2: return(high ); break;
      case  3: return(low  ); break;
      case  4: return((high + low)/2); break;
      case  5: return((high + low + close)/3); break;
      case  6: return((high + low + 2*close)/4); break;
      case  7: return((close + open)/2); break;
      case  8: return((high + low + close + open)/4); break;
      case  9: if(close > open) return((high + close)/2); else return((low + close)/2); break;
      case 10: if(close > open) return(high); else return(low); break;
      default: return(close); break;
   }
}

// HeikenAshi Price
double   haClose[2][2], haOpen[2][2], haHigh[2][2], haLow[2][2];
datetime prevhatime[2];

double HeikenAshi(int index,int price,int cbars,int bar)
{ 
   if(prevhatime[index] != Time[bar])
   {
      haClose[index][1] = haClose[index][0];
      haOpen [index][1] = haOpen [index][0];
      haHigh [index][1] = haHigh [index][0];
      haLow  [index][1] = haLow  [index][0];
      prevhatime[index] = Time[bar];
   }
   
   if(bar == cbars) 
   {
      haClose[index][0] = Close[bar];
      haOpen [index][0] = Open [bar];
      haHigh [index][0] = High [bar];
      haLow  [index][0] = Low  [bar];
   }
   else
   if(bar <  cbars)
   {
      haClose[index][0] = (Open[bar] + High[bar] + Low[bar] + Close[bar])/4;
      haOpen [index][0] = (haOpen[index][1] + haClose[index][1])/2;
      haHigh [index][0] = MathMax(High[bar],MathMax(haOpen[index][0],haClose[index][0]));
      haLow  [index][0] = MathMin(Low [bar],MathMin(haOpen[index][0],haClose[index][0]));
   }
   
   switch(price)
   {
      case  0: return(haClose[index][0]); break;
      case  1: return(haOpen [index][0]); break;
      case  2: return(haHigh [index][0]); break;
      case  3: return(haLow  [index][0]); break;
      case  4: return((haHigh [index][0] + haLow [index][0])/2); break;
      case  5: return((haHigh[index][0] + haLow[index][0] +   haClose[index][0])/3); break;
      case  6: return((haHigh[index][0] + haLow[index][0] + 2*haClose[index][0])/4); break;
      case  7: return((haClose[index][0] + haOpen[index][0])/2); break;
      case  8: return((haHigh[index][0] + haLow[index][0] +   haClose[index][0] + haOpen[index][0])/4); break;
      case  9: if(haClose[index][0] > haOpen[index][0]) return((haHigh[index][0] + haClose[index][0])/2); else return((haLow[index][0] + haClose[index][0])/2); break;
      case 10: if(haClose[index][0] > haOpen[index][0]) return(haHigh[index][0]); else return(haLow[index][0]); break;
      default: return(haClose[index][0]); break;
   }
} 